#ifndef __CRect__
#define __CRect__

#include "CPoint.hpp"
#include "CDimension.hpp"
#include "../Basics/CString.hpp"
using Exponent::Basics::CString;
using Exponent::MathTools::CDimension;
using Exponent::MathTools::CPoint;

//	===========================================================================

namespace Exponent
{
	namespace MathTools
	{
		/**
		 * @class CRect CRect.hpp
		 * @brief Rectangle object
		 *
		 * @date 19/09/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Added insert / offset functions
		 * @version 1.0.2 Added scaling functions
		 * @version 1.0.3 Added intersection functions
		 * @version 1.0.4 Added union function
		 * @version 1.0.5 Made the inline functions cross platform (@see FORCEINLINE)
		 * @version 1.0.6 Added getCentral!Position functions
		 * @version 1.0.7 Added dimension based functions, including c'tor
		 * @version 1.0.8 Added from string conversion
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CRect.hpp,v 1.4 2007/02/08 21:06:44 paul Exp $
		 */
		class CRect : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construciton
			 * @param left The left position
			 * @param top The top position
			 * @param width The width of the rectangle 
			 * @param height The height of the rectangle 
			 */
			CRect(const long left = 0, const long top = 0, const long width = 0, const long height = 0);

			/**
			 * Copy construction
			 * @param rect The rect to copy
			 */
			CRect(const CRect &rect);

			/**
			 * Construction
			 * @param origin The origin of the control
			 * @param dimension The dimensions of the rect
			 */
			CRect(const CPoint &origin, const CDimension &dimension);

			/**
			 * Destruction
			 */
			virtual ~CRect();

//	===========================================================================

			/**
			 * Assignment operator
			 * @param rect The rect to copy
			 * @retval CRect& A Reference to this
			 */
			virtual CRect &operator = (const CRect &rect);

			/**
			 * Equality operator
			 * @param rect The rect to compare to
			 * @retval bool True if rects are the same, false toherwise
			 */
			virtual bool operator == (const CRect &rect);

//	===========================================================================

			/**
			 * Set the entire rectangle
			 * @param left The left position
			 * @param top The top position
			 * @param width The width of the rectangle 
			 * @param height The height of the rectangle 
			 */
			void setRect(const long left, const long top, const long width, const long height);

			/**
			 * Set the rectangle
			 * @param origin The origin of the control
			 * @param dimension The dimensions of the rect
			 */
			void setRect(const CPoint &origin, const CDimension &dimension);

			/**
			 * Set the left
			 * @param left The left position
			 */
			void setLeft(const long left);

			/* 
			 * Set the width
			 * @param width The width of the rectangle 
			 */
			void setWidth(const long width);

			/**
			 * Set the top
			 * @param top The top position
			 */
			void setTop(const long top);

			/**
			 * Set the height
			 * @param height The height of the rectangle 
			 */
			void setHeight(const long height);

			/**
			 * Set the origin
			 * @param point The origin (left,top) point
			 */
			void setOrigin(const CPoint &point);

			/**
			 * Set the dimension
			 * @param dimension The dimensions of the rect
			 */
			void setDimensions(const CDimension &dimension);

//	===========================================================================

			/**
			 * Get the left
			 * @retval long The left
			 */
			long getLeft() const { return m_left; }

			/**
			 * Get the right
			 * @retval long The right
			 */
			long getRight() const { return m_right; }

			/**
			 * Get the top
			 * @retval long The top
			 */
			long getTop() const { return m_top; }

			/**
			 * Get the bottom
			 * @retval long The bottom
			 */
			long getBottom() const { return m_bottom; }

			/**
			 * Get the width
			 * @retval long The width
			 */
			long getWidth() const { return m_width; }

			/**
			 * Get the height
			 * @retval long The height
			 */
			long getHeight() const { return m_height; }

			/**
			 * Get the top left as a point
			 * @retval CPoint The top left as a point
			 */
			CPoint getOrigin() const { return CPoint(m_left, m_top); }

			/**
			 * Get the dimension as a point
			 * @retval CDimension The width and hight
			 */
			CDimension getDimension() const { return CDimension(m_width, m_height); }

			/**
			 * Get the central x position
			 * @retval long The central x position
			 */
			long getCentralXPosition() const;

			/**
			 * Get the central y position
			 * @retval long The central y position
			 */
			long getCentralYPosition() const;

//	===========================================================================

#ifdef WIN32
			/**
			 * Get as a windows RECT
			 * @retval RECT The widnows rect
			 */
			RECT getAsRect();

			/**
			 * Set from a window rect
			 * @param rect The windows rect to copy
			 */
			void setFromRect(RECT &rect);
#else
			/**
			 * Get as mac rect
			 * @retval Rect* The rectangle, you must dispose
			 */
			Rect *getAsRect() const;

			/**
			 * Set from a mac rect
			 * @param rect The carbon rect to copy
			 */
			void setFromRect(Rect &rect);
#endif

			/**
			 * Is the point inside the rectangle
			 * @param point The point to determine if its inside this rectangle
			 * @retval bool True if the point is inside this rectangle, false otherwise
			 */
			FORCEINLINE bool pointIsInside(const CPoint &point) const
			{
				return ((point.getXPosition() >= m_left)  &&
					   (point.getXPosition()  <= m_right) &&
					   (point.getYPosition()  >= m_top)   &&
					   (point.getYPosition()  <= m_bottom));
			}

			/**
			 * Is the rectangle completely inside this rectange
			 * @param rect The rectangle to determine if its inside this one
			 * @retval bool True if rect is completely inside this rectangle, false otherwise
			 */
			FORCEINLINE bool rectIsInside(const CRect &rect) const 
			{
				return ((rect.getLeft()   >= m_left)  &&
						(rect.getTop()    >= m_top)   &&
						(rect.getRight()  <= m_right) &&
						(rect.getBottom() <= m_bottom));
			}

			/**
			 * Does this rectangle interesect another?
			 * @param other The other rectangle to check if this intersects with
			 * @retval bool True if rectangles intersect, false otherwise
			 */
			FORCEINLINE bool rectanglesIntersect(const CRect& other) const
			{
				return !((m_left > other.getRight()) ||
						(m_top > other.getBottom())  ||
 						(other.getLeft() > m_right)  ||
						(other.getTop() > m_bottom));
			}

			/**
			 * Is this rectangle empty?
			 * @retval bool True if the rectangle is of size 0, false otherwise
			 */
			FORCEINLINE bool isRectangleEmpty() const
			{
				return (m_left == 0 && m_top == 0 && m_width == 0 && m_height == 0);
			}

			/**
			 * Offset the position of the rectangle
			 * @param point The amount to move the left and top by
			 */
			void offset(const CPoint &point);

			/**
			 * Inset the position of the rectangle (adds the values to the left and right, and removes it from the width)
			 * @param amount The amount to inset
			 */
			void inset(const long amount);

			/**
			 * Scale the rect (every value * by the value you give
			 * @param amount The amount to scale by
			 */
			void scale(const double amount);

			/**
			 * Get the intersection area of two rectangles
			 * @param rect1 Rectangle 1
			 * @param rect2 Rectangle 2
			 * @retval CRect* The intersection area of the two rectangles, you become the owner and responsible for deletion
			 */
			static CRect *getIntersectionArea(const CRect &rect1, const CRect &rect2);

			/**
			 * Get the intersection area of two rectangles
			 * @param rect1 Rectangle 1
			 * @param rect2 Rectangle 2
			 * @param output The intersection area of the two rectangles
			 */
			static void getIntersectionArea(const CRect &rect1, const CRect &rect2, CRect &output);
			
			/**
			 * Union this rectangle with another (creates rectangle the big enough to hold both)
			 * @param rect The rect to create a union with
			 */
			void unionWith(const CRect &rect);

			/**
			 * Get a description of the object
			 * @param string On return is filled with the description
			 * @param size The size of the stirng
			 */
			virtual void getObjectDescription(char *string, const long size) const;

			/**
			 * Convert from comma delimited string
			 * @param string The string to convert
			 * @retval CRect The new CRect, is 0 size rect if error
			 */
			static CRect convertFromString(const CString &string);

//	===========================================================================

		protected:

//	===========================================================================

			long m_width;					/**< Width of the rectangel */
			long m_height;					/**< Height of the rectangle */

			long m_left;					/**< Left position of the rectangle */
			long m_right;					/**< Right position of the rectangle */
			long m_top;						/**< Top position of the rectangle */
			long m_bottom;					/**< Bottom position of the rectangle */
		};
	}
}
#endif	// End of CRect.hpp